'use client';

import { useState } from 'react';
import Image from 'next/image';
import { useCart } from '@/hooks/use-cart';
import { useToast } from '@/hooks/use-toast';
import { useRestaurantInfo } from '@/hooks/use-restaurant-info';
import type { CartItem, Order, OrderItem } from '@/lib/types';
import { useFirestore, addDocumentNonBlocking } from '@/firebase';
import { collection, serverTimestamp } from 'firebase/firestore';


import { Button } from '@/components/ui/button';
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
  SheetFooter,
} from '@/components/ui/sheet';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Separator } from '@/components/ui/separator';
import { Input } from '@/components/ui/input';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import * as z from 'zod';
import {
  ShoppingBag,
  Plus,
  Minus,
  Trash2,
  ChevronLeft,
  Landmark,
  CreditCard,
  Wallet,
} from 'lucide-react';

const formSchema = z.object({
  name: z.string().min(2, { message: 'Nome deve ter pelo menos 2 caracteres.' }),
  address: z.string().min(5, { message: 'Endereço deve ter pelo menos 5 caracteres.' }),
  paymentMethod: z.enum(['Pix', 'Dinheiro', 'Cartão'], {
    required_error: 'Você precisa selecionar uma forma de pagamento.',
  }),
  changeFor: z.string().optional(),
});

type CheckoutFormValues = z.infer<typeof formSchema>;

const formatPrice = (price: number) => {
  return price.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
};

function generateWhatsAppMessage(
  customer: CheckoutFormValues,
  cart: CartItem[],
  total: number,
  restaurantInfo: any
) {

  let message = `Olá, meu nome é *${customer.name}*. Meu pedido é:\n\n`;
  message += `*${restaurantInfo.name}*\n`;
  message += `_Pedido mínimo:_ ${formatPrice(restaurantInfo.minOrder)}\n`;
  message += `_Cidade:_ ${restaurantInfo.city}\n`;
  message += `_Tempo de entrega:_ ${restaurantInfo.deliveryTime}\n\n`;
  message += '--- *ITENS DO PEDIDO* ---\n\n';

  cart.forEach(item => {
      const addonsPrice = item.selectedAddons.reduce((acc, addon) => acc + addon.price, 0);
      const itemPrice = item.product.price + addonsPrice;

      message += `  - ${item.quantity}x ${item.product.name} (${formatPrice(itemPrice)})\n`;
      if (item.selectedAddons.length > 0) {
          message += `    _Adicionais: ${item.selectedAddons.map(a => a.name).join(', ')}_\n`
      }
      message += `    _(${item.product.description})_\n`;
      if (item.product.imageUrl) message += `    _Imagem:_ ${item.product.imageUrl}\n`;
  });
  message += '\n';
  
  message += '--- *TOTAIS E PAGAMENTO* ---\n';
  message += `*Total do Pedido:* ${formatPrice(total)}\n`;
  message += `*Endereço de Entrega:* ${customer.address}\n`;
  message += `*Forma de Pagamento:* ${customer.paymentMethod}\n`;
  if (customer.paymentMethod === 'Dinheiro' && customer.changeFor) {
    message += `*Troco para:* R$ ${customer.changeFor}\n`;
  }
  
  message += `\nLink para WhatsApp: wa.me/${restaurantInfo.whatsappNumber}`;

  return encodeURIComponent(message);
}


export function CartSheet() {
  const [isOpen, setIsOpen] = useState(false);
  const [view, setView] = useState<'cart' | 'checkout'>('cart');
  const { cartItems, updateItemQuantity, removeItem, totalItems, totalPrice, clearCart } = useCart();
  const { restaurantInfo } = useRestaurantInfo();
  const { toast } = useToast();
  const firestore = useFirestore();

  const form = useForm<CheckoutFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: { name: '', address: '', changeFor: '' },
  });
  const paymentMethod = form.watch('paymentMethod');

  async function onSubmit(data: CheckoutFormValues) {
    // 1. Create order object
    const orderItems: OrderItem[] = cartItems.map(item => {
      const addonsPrice = item.selectedAddons.reduce((acc, addon) => acc + addon.price, 0);
      return {
        productId: item.product.id,
        productName: item.product.name,
        quantity: item.quantity,
        unitPrice: item.product.price + addonsPrice,
        addons: item.selectedAddons.map(a => ({ name: a.name, price: a.price })),
      }
    });

    const newOrder: Omit<Order, 'id'> = {
      customerName: data.name,
      deliveryAddress: data.address,
      paymentMethod: data.paymentMethod,
      orderDate: serverTimestamp(),
      totalAmount: totalPrice,
      orderItems: orderItems,
    };
    
    // 2. Save to Firestore
    if (firestore) {
      const ordersCollection = collection(firestore, 'orders');
      addDocumentNonBlocking(ordersCollection, newOrder);
    } else {
        toast({
            variant: "destructive",
            title: "Uh oh! Something went wrong.",
            description: "Could not connect to database.",
        });
        return;
    }


    // 3. Generate WhatsApp message and open link
    const message = generateWhatsAppMessage(data, cartItems, totalPrice, restaurantInfo);
    const whatsappUrl = `https://wa.me/${restaurantInfo.whatsappNumber}?text=${message}`;

    window.open(whatsappUrl, '_blank');
    
    toast({
      title: 'Pedido Enviado!',
      description: 'Seu pedido foi enviado e salvo com sucesso.',
    });
    
    // 4. Clean up
    form.reset();
    clearCart();
    setView('cart');
    setIsOpen(false);
  }

  const handleOpenChange = (open: boolean) => {
    if (!open) {
      setView('cart');
      form.reset();
    }
    setIsOpen(open);
  }

  const getItemSubtotal = (item: CartItem) => {
      const addonsPrice = item.selectedAddons.reduce((sum, addon) => sum + addon.price, 0);
      return (item.product.price + addonsPrice) * item.quantity;
  }

  return (
    <Sheet open={isOpen} onOpenChange={handleOpenChange}>
      <SheetTrigger asChild>
        <Button
          className="fixed bottom-6 right-6 h-16 w-16 rounded-full shadow-2xl z-40 bg-primary hover:bg-primary/90"
          size="icon"
          aria-label={`Abrir carrinho com ${totalItems} itens`}
        >
          <ShoppingBag className="h-7 w-7 text-primary-foreground" />
          {totalItems > 0 && (
            <span className="absolute -top-1 -right-1 flex h-6 w-6 items-center justify-center rounded-full bg-accent text-accent-foreground font-bold text-sm">
              {totalItems}
            </span>
          )}
        </Button>
      </SheetTrigger>
      <SheetContent className="flex flex-col p-0">
        {view === 'cart' && (
          <>
            <SheetHeader className="p-6">
              <SheetTitle className="text-2xl font-headline">Sua Sacola</SheetTitle>
            </SheetHeader>
            {cartItems.length > 0 ? (
              <>
                <ScrollArea className="flex-grow px-6">
                  <div className="flex flex-col gap-6">
                    {cartItems.map((item) => (
                      <div key={item.id} className="flex gap-4">
                        <Image
                          src={item.product.imageUrl}
                          alt={item.product.name}
                          width={80}
                          height={80}
                          className="rounded-md object-cover"
                        />
                        <div className="flex-grow">
                          <h3 className="font-semibold">{item.product.name}</h3>
                           {item.selectedAddons.length > 0 && (
                                <p className="text-xs text-muted-foreground">
                                    + {item.selectedAddons.map(a => a.name).join(', ')}
                                </p>
                            )}
                          <p className="text-sm text-primary font-semibold">{formatPrice(getItemSubtotal(item))}</p>
                          <div className="flex items-center gap-2 mt-2">
                            <Button variant="outline" size="icon" className="h-7 w-7" onClick={() => updateItemQuantity(item.id, item.quantity - 1)}><Minus className="h-4 w-4" /></Button>
                            <span className="w-6 text-center">{item.quantity}</span>
                            <Button variant="outline" size="icon" className="h-7 w-7" onClick={() => updateItemQuantity(item.id, item.quantity + 1)}><Plus className="h-4 w-4" /></Button>
                          </div>
                        </div>
                        <Button variant="ghost" size="icon" className="text-muted-foreground" onClick={() => removeItem(item.id)}><Trash2 className="h-4 w-4" /></Button>
                      </div>
                    ))}
                  </div>
                </ScrollArea>
                <SheetFooter className="p-6 mt-auto bg-card border-t">
                  <div className="w-full space-y-4">
                    <div className="flex justify-between font-bold text-lg">
                      <span>Total</span>
                      <span>{formatPrice(totalPrice)}</span>
                    </div>
                    <Button className="w-full" size="lg" onClick={() => setView('checkout')}>Continuar</Button>
                  </div>
                </SheetFooter>
              </>
            ) : (
              <div className="flex flex-col items-center justify-center h-full text-center p-6">
                <ShoppingBag className="h-20 w-20 text-muted-foreground/50 mb-4" />
                <h3 className="text-xl font-semibold">Sua sacola está vazia</h3>
                <p className="text-muted-foreground">Adicione itens do cardápio para começar.</p>
              </div>
            )}
          </>
        )}

        {view === 'checkout' && (
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col h-full">
              <SheetHeader className="p-6 flex-row items-center gap-4">
                <Button variant="ghost" size="icon" onClick={() => setView('cart')}><ChevronLeft className="h-5 w-5" /></Button>
                <SheetTitle className="text-2xl font-headline">Finalizar Pedido</SheetTitle>
              </SheetHeader>
              <ScrollArea className="flex-grow px-6">
                <div className="space-y-6">
                   <FormField control={form.control} name="name" render={({ field }) => (
                      <FormItem>
                        <FormLabel>Seu Nome</FormLabel>
                        <FormControl><Input placeholder="John Doe" {...field} /></FormControl>
                        <FormMessage />
                      </FormItem>
                    )} />
                    <FormField control={form.control} name="address" render={({ field }) => (
                      <FormItem>
                        <FormLabel>Endereço de Entrega</FormLabel>
                        <FormControl><Input placeholder="Rua, Número, Bairro, Cidade" {...field} /></FormControl>
                        <FormMessage />
                      </FormItem>
                    )} />
                    <FormField control={form.control} name="paymentMethod" render={({ field }) => (
                      <FormItem className="space-y-3">
                        <FormLabel>Forma de Pagamento</FormLabel>
                        <FormControl>
                          <RadioGroup onValueChange={field.onChange} defaultValue={field.value} className="flex flex-col space-y-1">
                            <FormItem className="flex items-center space-x-3 space-y-0">
                              <FormControl><RadioGroupItem value="Pix" /></FormControl>
                              <FormLabel className="font-normal flex items-center gap-2"><Landmark className="h-4 w-4 text-muted-foreground" /> Pix</FormLabel>
                            </FormItem>
                            <FormItem className="flex items-center space-x-3 space-y-0">
                              <FormControl><RadioGroupItem value="Dinheiro" /></FormControl>
                              <FormLabel className="font-normal flex items-center gap-2"><Wallet className="h-4 w-4 text-muted-foreground" /> Dinheiro</FormLabel>
                            </FormItem>
                             <FormItem className="flex items-center space-x-3 space-y-0">
                              <FormControl><RadioGroupItem value="Cartão" /></FormControl>
                              <FormLabel className="font-normal flex items-center gap-2"><CreditCard className="h-4 w-4 text-muted-foreground" /> Cartão</FormLabel>
                            </FormItem>
                          </RadioGroup>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )} />
                    {paymentMethod === 'Dinheiro' && (
                      <FormField control={form.control} name="changeFor" render={({ field }) => (
                        <FormItem>
                          <FormLabel>Troco para (opcional)</FormLabel>
                          <FormControl><Input type="number" placeholder="50.00" {...field} /></FormControl>
                          <FormMessage />
                        </FormItem>
                      )} />
                    )}
                </div>
              </ScrollArea>
              <SheetFooter className="p-6 mt-auto bg-card border-t">
                  <div className="w-full space-y-4">
                    <div className="flex justify-between font-bold text-lg">
                      <span>Total</span>
                      <span>{formatPrice(totalPrice)}</span>
                    </div>
                    <Button type="submit" className="w-full" size="lg">Enviar pedido via WhatsApp</Button>
                  </div>
              </SheetFooter>
            </form>
          </Form>
        )}
      </SheetContent>
    </Sheet>
  );
}
